import tkinter as tk
from tkinter import filedialog, messagebox, simpledialog
from PIL import Image, ImageTk
import time
import csv
import webbrowser
import json
from matplotlib import pyplot as plt

class RiderTimerApp:
    def __init__(self, root):
        self.root = root
        self.root.title("Motocross Rider Timer")

        self.sponsor_logo_path = None
        self.sponsor_url = "http://www.freeusbapps.com"  # Pre-set the URL to www.freeusbapps.com

        # Sponsor Banner Section (Image and Text as Hyperlink)
        self.sponsor_banner_frame = tk.Frame(root)
        self.sponsor_banner_frame.pack(pady=5)

        self.sponsor_logo_label = tk.Label(self.sponsor_banner_frame)
        self.sponsor_logo_label.pack(side=tk.LEFT)

        self.sponsor_text_label = tk.Label(self.sponsor_banner_frame, text="Click here for sponsor", font=("Helvetica", 12), fg="blue", cursor="hand2")
        self.sponsor_text_label.pack(side=tk.LEFT, padx=10)

        self.upload_race_sponsor_btn = tk.Button(self.sponsor_banner_frame, text="Upload Race Sponsor Logo", command=self.upload_race_sponsor_logo)
        self.upload_race_sponsor_btn.pack(pady=5)

        # Bind the sponsor text label to open the URL when clicked
        self.sponsor_text_label.bind("<Button-1>", self.open_sponsor_url)

        # Bind the sponsor logo label to open the URL when clicked
        self.sponsor_logo_label.bind("<Button-1>", self.open_sponsor_url)

        # Meta Information Section
        self.meta_frame = tk.Frame(root)
        self.meta_frame.pack(pady=5)

        tk.Label(self.meta_frame, text="Race Name:").grid(row=0, column=0)
        self.race_name = tk.Entry(self.meta_frame, width=20)
        self.race_name.grid(row=0, column=1)

        tk.Label(self.meta_frame, text="Location:").grid(row=0, column=2)
        self.location = tk.Entry(self.meta_frame, width=20)
        self.location.grid(row=0, column=3)

        tk.Label(self.meta_frame, text="Date:").grid(row=0, column=4)
        self.date = tk.Entry(self.meta_frame, width=15)
        self.date.grid(row=0, column=5)

        # Rider Setup Section
        self.setup_frame = tk.Frame(root)
        self.setup_frame.pack(pady=10)

        tk.Label(self.setup_frame, text="Number of Riders:").pack(side=tk.LEFT)
        self.rider_entry = tk.Entry(self.setup_frame, width=5)
        self.rider_entry.pack(side=tk.LEFT, padx=5)

        self.create_button = tk.Button(self.setup_frame, text="Create Riders", command=self.create_riders)
        self.create_button.pack(side=tk.LEFT, padx=5)

        self.start_all_button = tk.Button(self.setup_frame, text="Start All", command=self.start_all_timers)
        self.start_all_button.pack(side=tk.LEFT, padx=5)

        self.save_button = tk.Button(self.setup_frame, text="Save Session", command=self.save_session)
        self.save_button.pack(side=tk.LEFT, padx=5)

        self.load_button = tk.Button(self.setup_frame, text="Load Session", command=self.load_session)
        self.load_button.pack(side=tk.LEFT, padx=5)

        self.export_button = tk.Button(self.setup_frame, text="Export to CSV", command=self.export_to_csv)
        self.export_button.pack(side=tk.LEFT, padx=5)

        self.leaderboard_button = tk.Button(self.setup_frame, text="Show Leaderboard", command=self.show_leaderboard)
        self.leaderboard_button.pack(side=tk.LEFT, padx=5)

        self.rider_widgets = []
        self.riders_frame = tk.Frame(root)
        self.riders_frame.pack(pady=5)

        self.league_file = "league_results.json"
        self.race_counter = 1

    def upload_race_sponsor_logo(self):
        path = filedialog.askopenfilename(filetypes=[("Images", "*.png;*.jpg;*.jpeg;*.gif")])
        if path:
            # Open and resize the image
            img = Image.open(path)
            img = img.resize((200, 80), Image.Resampling.LANCZOS)
            img = ImageTk.PhotoImage(img)
            self.sponsor_logo_label.config(image=img)
            self.sponsor_logo_label.image = img
            self.sponsor_logo_path = path

            # Ask for the sponsor's domain URL
            self.sponsor_url = "http://www.freeusbapps.com"  # We set this to the fixed URL
            self.sponsor_text_label.config(text=self.sponsor_url)

    def open_sponsor_url(self, event):
        webbrowser.open(self.sponsor_url)

    def create_riders(self):
        try:
            num_riders = int(self.rider_entry.get())
        except ValueError:
            messagebox.showerror("Error", "Please enter a valid number of riders.")
            return

        # Clear previous rider widgets if any
        for widget in self.riders_frame.winfo_children():
            widget.destroy()
        self.rider_widgets.clear()

        for i in range(num_riders):
            self.create_rider_row(i)

    def create_rider_row(self, index):
        frame = tk.Frame(self.riders_frame, relief=tk.RIDGE, borderwidth=2, padx=10, pady=10)
        frame.pack(pady=5, fill=tk.X)

        tk.Label(frame, text=f"Rider {index + 1}", font=("Helvetica", 12, "bold")).grid(row=0, column=0, padx=5, sticky='w')

        name_entry = tk.Entry(frame, width=15)
        number_entry = tk.Entry(frame, width=8)
        bike_entry = tk.Entry(frame, width=12)

        name_entry.insert(0, f"Rider{index+1}")
        number_entry.insert(0, f"#{index+10}")
        bike_entry.insert(0, "Yamaha")

        name_entry.grid(row=1, column=1)
        number_entry.grid(row=1, column=2)
        bike_entry.grid(row=1, column=3)

        passport_label = tk.Label(frame)
        passport_label.grid(row=0, column=4, rowspan=2, padx=5)
        passport_btn = tk.Button(frame, text="Upload Photo", command=lambda: self.upload_image(passport_label, index, 'passport'))
        passport_btn.grid(row=1, column=4, sticky='s')

        logo_label = tk.Label(frame)
        logo_label.grid(row=0, column=5, rowspan=2, padx=5)
        upload_btn = tk.Button(frame, text="Upload Logo", command=lambda: self.upload_image(logo_label, index, 'logo'))
        upload_btn.grid(row=1, column=5, sticky='s')

        time_label = tk.Label(frame, text="00:00.0", font=("Courier", 14))
        time_label.grid(row=0, column=6, padx=10)

        start_btn = tk.Button(frame, text="Start", command=lambda: self.start_timer(index))
        pause_btn = tk.Button(frame, text="Pause", command=lambda: self.pause_timer(index))
        resume_btn = tk.Button(frame, text="Resume", command=lambda: self.resume_timer(index))
        stop_btn = tk.Button(frame, text="Stop", command=lambda: self.stop_timer(index))
        lap_btn = tk.Button(frame, text="Lap", command=lambda: self.record_lap(index))
        graph_btn = tk.Button(frame, text="Graph", command=lambda: self.show_graph(index))

        start_btn.grid(row=1, column=6)
        pause_btn.grid(row=1, column=7)
        resume_btn.grid(row=1, column=8)
        stop_btn.grid(row=1, column=9)
        lap_btn.grid(row=1, column=10)
        graph_btn.grid(row=1, column=11)

        lap_listbox = tk.Listbox(frame, height=4, width=20)
        lap_listbox.grid(row=0, column=7, rowspan=1, columnspan=2)

        self.rider_widgets.append({
            "name": name_entry,
            "number": number_entry,
            "bike": bike_entry,
            "logo_label": logo_label,
            "logo_path": None,
            "passport_label": passport_label,
            "passport_path": None,
            "time_label": time_label,
            "start_time": 0,
            "paused_time": 0,
            "running": False,
            "laps": [],
            "lap_listbox": lap_listbox,
            "timer_job": None
        })

    def upload_image(self, label, index, image_type):
        path = filedialog.askopenfilename(filetypes=[("Images", "*.png;*.jpg;*.jpeg;*.gif")])
        if path:
            img = Image.open(path)
            img = img.resize((120, 60), Image.Resampling.LANCZOS) if image_type == 'logo' else img.resize((60, 80), Image.Resampling.LANCZOS)
            img = ImageTk.PhotoImage(img)
            label.config(image=img)
            label.image = img
            self.rider_widgets[index][f"{image_type}_path"] = path

    def start_timer(self, index):
        rider = self.rider_widgets[index]
        if not rider['running']:
            rider['start_time'] = time.time() - rider['paused_time']  # Adjust for any previous pause time
            rider['running'] = True
            self.update_timer(index)

    def update_timer(self, index):
        rider = self.rider_widgets[index]
        if rider['running']:
            elapsed = time.time() - rider['start_time']
            mins = int(elapsed // 60)
            secs = elapsed % 60
            rider['time_label'].config(text=f"{mins:02d}:{secs:04.1f}")
            rider['timer_job'] = self.root.after(100, lambda: self.update_timer(index))

    def pause_timer(self, index):
        rider = self.rider_widgets[index]
        if rider['running']:
            rider['paused_time'] = time.time() - rider['start_time']
            rider['running'] = False
            if rider['timer_job']:
                self.root.after_cancel(rider['timer_job'])

    def resume_timer(self, index):
        self.start_timer(index)

    def stop_timer(self, index):
        rider = self.rider_widgets[index]
        rider['running'] = False
        if rider['timer_job']:
            self.root.after_cancel(rider['timer_job'])
        # The total time is not reset, it remains as the last recorded time
        rider['paused_time'] = time.time() - rider['start_time']  # Store the time when stopped
        # Time is displayed and will remain on the label until resumed or reset.

    def record_lap(self, index):
        rider = self.rider_widgets[index]
        if rider['running']:
            lap_time = time.time() - rider['start_time']
            rider['laps'].append(lap_time)
            rider['lap_listbox'].insert(tk.END, f"Lap {len(rider['laps'])}: {lap_time:.2f}s")

    def show_graph(self, index):
        rider = self.rider_widgets[index]
        if not rider['laps']:
            messagebox.showinfo("No Data", "No lap data available to graph.")
            return
        plt.plot(rider['laps'], marker='o')
        plt.title(f"Lap Times for {rider['name'].get()}")
        plt.xlabel("Lap")
        plt.ylabel("Time (s)")
        plt.grid(True)
        plt.show()

    def start_all_timers(self):
        for i in range(len(self.rider_widgets)):
            self.start_timer(i)

    def show_leaderboard(self):
        leaderboard = []
        for rider in self.rider_widgets:
            name = rider['name'].get()
            laps = rider['laps']
            if laps:
                avg_time = sum(laps) / len(laps)
                leaderboard.append((name, avg_time))

        leaderboard.sort(key=lambda x: x[1])

        top = tk.Toplevel(self.root)
        top.title("Leaderboard - Average Lap Time")

        for i, (name, avg_time) in enumerate(leaderboard):
            tk.Label(top, text=f"{i+1}. {name}: {avg_time:.2f}s").pack()

    def save_session(self):
        filename = filedialog.asksaveasfilename(defaultextension=".csv", filetypes=[("CSV files", "*.csv")])
        if filename:
            with open(filename, mode="w", newline="") as file:
                writer = csv.writer(file)

                writer.writerow(["Rider Name", "Rider Number", "Bike Model", "Lap Times"])

                for rider in self.rider_widgets:
                    name = rider['name'].get()
                    number = rider['number'].get()
                    bike = rider['bike'].get()
                    laps = rider['laps']
                    lap_times = ', '.join([f"{lap:.2f}" for lap in laps])

                    writer.writerow([name, number, bike, lap_times])

            messagebox.showinfo("Success", "Session saved successfully!")

    def load_session(self):
        pass

    def export_to_csv(self):
        pass

if __name__ == "__main__":
    root = tk.Tk()
    app = RiderTimerApp(root)
    root.mainloop()
